/************************************************************************
* \file: trace_sharedmem.c
*
* \version: $Id: trace_sharedmem.c,v 1.8 2012/05/28 10:30:38 jayanth.mc Exp $
*
* This file implements Trace specific Shared memory wrappers 
*
* \component: Gen2 Trace
*
* \author   Sakthivelu S.  sakthivelu.s@in.bosch.com
*           Diezel, Eckhard (ITUC; ADITG/SWG) [ediezel@de.adit-jv.com]
* \copyright: (c) 2003 - 2009 ADIT
*
***********************************************************************/

#include <grp.h>
#include "trace_base.h"

/*
 * All trace application users/groups will be added under trace-be group
 * for trace application and its shared memory access
 */
#define TRACE_GROUP_NAME    "trace-be"

#define TRACE_ACCESS_RIGTHS_RW       S_IRUSR | S_IWUSR | S_IRGRP | S_IWGRP /* 0660 */

/**
 * Create/attach to Shared Memory
 *
 * \param sh    Pointer to trace sharedmem type
 * \param name  name of the posix shared mem
 * \param size  size of the shared mem
 *
 * \return ER   E_OK    if successful
 *              E_FAIL  otherwise
 */
EXPORT ER TRACE_sharedmem_create(TRACE_sharedmem* sh, const S8* name, size_t size)
{
	S32 shm_fd;
	mode_t mode;
	gid_t TraceGroupId = 0;

	if ((NULL == sh) || (NULL == name))
	{
		return E_FAIL;
	}
	sh->size = 0;
	sh->mem = NULL;

	if ((sh->name= (S8*)strdup((char*)name)) == NULL)
	{
		return E_FAIL;
	}

	/* eco_osal Read trace group name and properties */
	struct group *pTraceGrp = getgrnam(TRACE_GROUP_NAME);
	if(pTraceGrp==NULL)
	{
		TRACE_SYSLOG(TRACE_SYSLOG_ERROR, "getgrnam -> error %d \n", errno);
	}
	else
	{
		TraceGroupId = pTraceGrp->gr_gid;
		//TRACE_SYSLOG(TRACE_SYSLOG_ERROR, "TRACE_sharedmem_create pid = %d gid = %d  TraceGroupId = %d\n", getpid(),getgid(),TraceGroupId);
	}
	/*
	 * first grab the semaphore to protect the memory
	 * area in case we need to initialize it
	 * initialize it empty so all others block on it
	 * (sem_open() is atomic for the calling processes
	 *  so this will always work)
	 */
	sh->bNew = TRUE;
   /* @kbm2kor: Fix:SWGIIX-112
    * set process umask to 0 temporarily
    * ADTRCsh should have permission 777 for normal users
    *
    * @kap3kor start: Fix:RTC-251450 and NCG3D-88744
    * ADTRCsh permission have changed to 0666(RW) for normal users
    * to avoid world-writable permissions for trace shared memory.
    * Read-Write permission to USER | GROUP | OWNER with below options
    * [S_IRUSR | S_IWUSR | S_IRGRP | S_IWGRP | S_IROTH | S_IWOTH]
    * @kap3kor end:
    *
    * file permission are set as (mode & (~umask))
    */
   mode = umask(0);
	if ((sh->sem = (sem_t*)sem_open((char*)sh->name, (O_CREAT | O_EXCL), TRACE_ACCESS_RIGTHS_RW, 0))
					== SEM_FAILED)
	{
		/*
		 *  semaphore already exists
		 */
		sh->bNew = FALSE;
		if ((sh->sem = sem_open((char*)sh->name, 0)) == SEM_FAILED)
		{
			return E_FAIL;
		}
		/*
		 * wait until the first process is done initializing
		 * the shared memory area
		 */
		sem_wait(sh->sem);
		sem_post(sh->sem);
	}

	if(TraceGroupId)
	{
		if(chown("/dev/shm/sem.ADTRCsh",(uid_t)-1,TraceGroupId) == -1)
		{
			TRACE_SYSLOG(TRACE_SYSLOG_ERROR, "/dev/shm/sem.ADTRCsh chown error %d TraceGroupId  %d\n", errno, TraceGroupId);
		}
		if(chmod("/dev/shm/sem.ADTRCsh", TRACE_ACCESS_RIGTHS_RW) == -1)
		{
			TRACE_SYSLOG(TRACE_SYSLOG_ERROR, "/dev/shm/sem.ADTRCsh chmod error %d\n", errno);
		}
	}

	/* Try to open the shm instance normally and share it with
	 * existing clients
	 */
	if((shm_fd = shm_open((char*)sh->name, (O_CREAT | O_RDWR), TRACE_ACCESS_RIGTHS_RW)) < 0)
	{
		return E_FAIL;
	}

	if(TraceGroupId)
	{
		/* adapt rights for shared memory */
		if(fchown(shm_fd,(uid_t)-1,TraceGroupId) == -1)
		{
			//TRACE_SYSLOG(TRACE_SYSLOG_ERROR, "Trace shm fchown error %d \n",errno);
		}
		if(fchmod(shm_fd, TRACE_ACCESS_RIGTHS_RW) == -1)
		{
			//TRACE_SYSLOG(TRACE_SYSLOG_ERROR, "Trace shm fchmod error %d \n",errno);
		}
	}

	/* Set the size of the SHM */
	if (ftruncate(shm_fd, size) != 0)
	{
		return E_FAIL;
	}
	sh->size = size;

	/* Connect the mem pointer to set to the shared memory area,
	 * with desired permissions
	 */
	if((sh->mem =  mmap(0, sh->size, (PROT_READ | PROT_WRITE),
	                   MAP_SHARED, shm_fd, 0)) == MAP_FAILED)
	{
	    return E_FAIL;
	}
	/*
	 * fd is not needed any more
	 */
	(void)close(shm_fd);
        
  /* set back to original process umask */
  mode = umask(mode);

	return E_OK;
}

/**
 * Destroy/detach to Shared Memory
 *
 * \param sh      Pointer to trace sharedmem type
 * \param unlink  flag indicating unlink or only close
 *
 * \return ER   E_OK    if successful
 *              E_FAIL  otherwise
 */
EXPORT ER TRACE_sharedmem_destroy(TRACE_sharedmem *sh, BOOL un_link)
{
	if (NULL == sh)
	{
		return E_FAIL;
	}

	if (sh->mem != NULL)
	{
		(void)munmap(sh->mem, sh->size);
	}

	(void)sem_close(sh->sem);

	if (sh->name != NULL)
	{
		if (un_link)
		{
			(void)shm_unlink((char*)sh->name);
			(void)sem_unlink((char*)sh->name);
		}
		free(sh->name);
		sh->name = NULL;
	}
	sh->size = 0;
	sh->mem = NULL;
	return E_OK;
}

/**
 * Get whether Shared memory is created or attached
 *
 * \param sh      Pointer to trace sharedmem type
 *
 * \return ER   TRUE    if created newly
 *              FALSE   if attached to already existing one
 */
EXPORT BOOL TRACE_sharedmem_isnew(TRACE_sharedmem *sh)
{
	if (NULL == sh)
	{
		return FALSE;
	}
	return sh->bNew;
}

/**
 * Signals other processes once init is done
 *
 * \param sh      Pointer to trace sharedmem type
 *
 * \return none
 */
EXPORT void TRACE_sharedmem_init_done(TRACE_sharedmem *sh)
{
	if (NULL == sh)
	{
		return;
	}
	sh->bNew = FALSE;
	sem_post(sh->sem);
}

/**
 * Gets sharedmem address
 *
 * \param sh      Pointer to trace sharedmem type
 *
 * \return ptr to shared memory start addr
 */
EXPORT void* TRACE_sharedmem_mem(TRACE_sharedmem *sh)
{
	if (NULL == sh)
	{
		return NULL;
	}
	return sh->mem;
}
